﻿/*  Author:     Ben Hall
 *  File:       BalancedAI.cs
 *  Date:       4/28/2011
 *  
 *  Purpose:    BalancedAI.cs extends the AICharacter class and overrides the abstract method MakeMove()
 *              using a decision tree that has a 50/50 chance of choosing an offensive or defensive action.
 *              When it selects an offensive action, it tries to use its strongest attack on the player,
 *              inflict negative status effects, or use the Fight action.
 *              When using a defensive action, it chooses between healing itself or its allies, adding
 *              positive status effects to itself or allies, or may just Defend.
 *              
 *              The MakeMove() method also uses a random chance to skip the "best" move the AICharacter
 *              can make at each branch, based on the AICharacter's intelligence, with the chance of it
 *              skipping a "best" move decreasing every time the "best" move is skipped.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace OnlineGame
{
    public class BalancedAI : AICharacter
    {
        public BalancedAI(String name, int strIn, int stmIn, int aglIn, int intIn, int fthIn, int prsIn, int currencyIn)
        {
            this.characterName = name;
            this.strength = new CharStat(strIn);
            this.stamina = new CharStat(stmIn);
            this.agility = new CharStat(aglIn);
            this.intelligence = new CharStat(intIn);
            this.faith = new CharStat(fthIn);
            this.persona = new CharStat(prsIn);
            //this.level = DetermineLevel();

            this.currencyCarried = currencyIn;

            this.maxHealth = DetermineMaxHealth();
            this.currentHealth = maxHealth;
            this.maxMagic = DetermineMaxMagic();
            this.currentMagic = maxMagic;

            this.defending = false;
            this.poisoned = false;
            this.covering = false;
            this.coveredBy = null;
            this.characterCovered = null;

            this.skillsKnown = new bool[MainPage.skillList.Length];
            for (int i = 0; i < skillsKnown.Length; i++)
            {
                skillsKnown[i] = false;
            }

            this.decisionChance = DetermineDecisionChance();

        }

        /*  MakeMove() selects an action for the AICharacter to take, using a decision tree with a
         *  random chance of skipping the "best move" at each branch (based on the AICharacter's intelligence
         *  score)
         *  
         *  BalancedAI has no specific priority, it will decide between either an offensive action against
         *  its opponent, or a defensive action to assist itself or its allies.
         */
        public override String MakeMove()
        {
            String messageReturned = "";
            this.decisionChance = DetermineDecisionChance();
            this.rndDecision = MainPage.rnd.NextDouble();
            //AI may decide to skip more optimal moves, with lower intelligence being more likely to skip moves
            bool useBestMove = false;
            useBestMove = (rndDecision < decisionChance);

            int chosenSkill = GetCompHealSkill();
            if ((this.GetHealthPercent() < lowHealth) && (chosenSkill >= 0) && (useBestMove))
            {
                //heal self if at low health
                messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, this);
            }
            else
            {
                if (!useBestMove)
                {
                    useBestMove = RedetermineUseBestMove();
                }
                //50% chance of offensive action, 50% chance of defensive action
                if (MainPage.rnd.NextDouble() < 0.5)
                {
                    //offensive action
                    //30% chance of debuff action, 70% chance of best attack
                    if (MainPage.rnd.NextDouble() < 0.3)
                    {
                        messageReturned = UseRandomDebuff();
                    }
                    else
                    {
                        messageReturned = UseBestAttack();
                    }
                }
                else
                {
                    //defensive action
                    chosenSkill = GetCompHealSkill();
                    AICharacter healTarget = null;
                    //find target under 25% health for healing.
                    for (int i = 0; i < MainPage.compTeam.Length; i++)
                    {
                        if ((MainPage.compTeam[i].GetCurrentHealth() > 0) && (!MainPage.compTeam[i].Equals(this)) && (MainPage.compTeam[i].GetHealthPercent() < lowHealth))
                        {
                            if (healTarget == null)
                            {
                                healTarget = MainPage.compTeam[i];
                            }
                            else if (healTarget.GetCurrentHealth() > MainPage.compTeam[i].GetCurrentHealth())
                            {
                                healTarget = MainPage.compTeam[i];
                            }
                        }
                    }
                    if ((healTarget != null) && (chosenSkill >= 0) && (useBestMove))
                    {
                        //use healing skill on targeted ally
                        messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, healTarget);
                    }
                    else
                    {
                        //redetermine useBestMove
                        if (!useBestMove)
                        {
                            useBestMove = RedetermineUseBestMove();
                        }
                        //60% chance of general Defensive action, 40% chance of other actions
                        if (MainPage.rnd.NextDouble() < 0.6)
                        {
                            messageReturned = UseBestDefense();
                        }
                        else
                        {
                            //if poisoned, heal self
                            chosenSkill = GetCompRemovePoisonSkill();
                            if ((this.IsPoisoned()) && (chosenSkill >= 0) && (useBestMove))
                            {
                                //heal self of poison
                                messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, this);
                            }
                            else
                            {
                                //redetermine useBestMove
                                if (!useBestMove)
                                {
                                    useBestMove = RedetermineUseBestMove();
                                }

                                AICharacter coverTarget = null;
                                //find target under 25% health for healing.
                                for (int i = 0; i < MainPage.compTeam.Length; i++)
                                {
                                    if ((MainPage.compTeam[i].GetCurrentHealth() > 0) && (!MainPage.compTeam[i].Equals(this)) && (MainPage.compTeam[i].GetHealthPercent() < lowHealth))
                                    {
                                        if (coverTarget == null)
                                        {
                                            coverTarget = MainPage.compTeam[i];
                                        }
                                        else if (coverTarget.GetCurrentHealth() > MainPage.compTeam[i].GetCurrentHealth())
                                        {
                                            coverTarget = MainPage.compTeam[i];
                                        }
                                    }
                                }

                                if ((coverTarget != null) && (this.CanUseSkill(GetSkillWithName("Cover"))) && (!this.IsCovered()) && (!this.IsCovering()) && (useBestMove))
                                {
                                    //cover wounded ally
                                    messageReturned = MainPage.UseSkill(MainPage.skillList[GetSkillWithName("Cover")], this, coverTarget);
                                }
                                else
                                {
                                    if (!useBestMove)
                                    {
                                        useBestMove = RedetermineUseBestMove();
                                    }
                                    //if self is under 50% health, heal self
                                    chosenSkill = GetCompHealSkill();
                                    if ((GetHealthPercent() < halfHealth) && (chosenSkill >= 0) && (useBestMove))
                                    {
                                        //heal self
                                        messageReturned = MainPage.UseSkill(MainPage.skillList[chosenSkill], this, this);
                                    }
                                    else
                                    {
                                        //80% chance of defensive action, 20% chance of attack action
                                        if (MainPage.rnd.NextDouble() < 0.8)
                                        {
                                            messageReturned = UseBestDefense();
                                        }
                                        else
                                        {
                                            messageReturned = UseBestAttack();
                                        }
                                    }
                                }
                            }
                        }

                    }
                }
            }
            return messageReturned;
        }

    }
}
